/****************************************************************
*	                                                            *
*  Steuerprogramm fr 2m Funkmodul DRA818V	LCD Anzeigeversion	*
* 									    						*
*  Meinrad Gtz                                                 *
*                                                               *
*  Version 1.0                 17. August 2017	                *
*                                                               *
*  Programmbeschreibung:                                        *
*                                                               *
*									    						*
*  Ansteuerung eines Funkmoduls DRA818V (2m) ber die Serielle	*
*  Schnittstelle. 							    				*
* (9600 Baud 1 Start Bit, 8 Data Bits, 1 Stop Bit, No Parity)	*
*									    						*
*	Funktionsumfang:				  							*
* 	Frequenzwahl im Bereich 144 -146 MHz						*
*	Ablage 0/600kHz												*
*	Kanalbreite 12,5kHz/25kHz									*
*	Squelch: 0 - 8	(0 = Spuelch Off)							*
*	Volumen: 1 - 8												*
*	Sendeleistung: 0,5W/1,0W									*
*																*
*	Relaiston 1750 Hz											*
*																*
*	Funktionsanzeige auf einem zweizeiligen LCD					*
*									    						*
*									    						*
*  Prozessor:		ATMEGA 8L						    		*
*  Taktfrequenz:	3,686 MHz					    			*
*                                                               *
****************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

//Defines
#define TIMER0_10MS		144		//Ladewert Timer 0 fr 10ms
#define TIMER0_1750		127 	//Ladewert Timer 0 fr 1750 Hz

#define ENTPRELLZEIT  4			//Tasten Entprellzeit = TIMER0_10MS * ENTPRELLZEIT
#define TASTENMASKE	  0b11111100;
#define ANZ_VERS_NR   3			//Anzeigedauer Versionsnummer in Sekunden
#define ANZ_ERROR	  5			//Anzeigedauer Fehlermeldung in Sekungen

#define POWER		PORTB0	//Power Auswahlleitung (0 -> 0,5W; 1 -> 1,0W)
#define PTT			PORTB1	//PTT-Leitung (Taste)
#define TON_1750	PORTB2	//1750 Hz Signalausgang

#define TRUE		1
#define FALSE		0

#define UHF_RX		1
#define UHF_TX		0

#define TON			0
#define TIMER		1


//Function declaration
void Init_Timer0(void);
void Convert_US_Int_Bcd(unsigned long int Zahl);
void Convert_BCD_ASCII(void);
void Init_CPU(void);
void Init_LCD(void);
void LCD_Zeile1(unsigned char *pLCD);
void LCD_Zeile2(unsigned char *pLCD);
void Show_Frequenz_Zeile2(unsigned long int uliFrequenz);
void WR_COMAND(unsigned char i);
void WR_DATA(unsigned char i);
void Wait(unsigned int i);
void Pause_Timer (unsigned char Zeit);
void Wait_Tasten_Open(void);
void Funktions_Auswahl(void);

void Init_UART(void);
void Send_CR_LF (void);
void Send_String (unsigned char *pSource, unsigned char cAnzahl);
void Receive_String (unsigned char *pDestination, unsigned char cAnzahl);

void Init_DRA818 (void);
unsigned char Send_SetGroup(unsigned char cBW, unsigned long int uliTx_Frequenz, unsigned long int uliRx_Frequenz,
				   			unsigned int Tx_CTCSS, unsigned char cSqu, unsigned int Rx_CTCSS);
unsigned char Send_VolumeCommand(unsigned char cVol);
unsigned char Send_FilterCommand(unsigned char cDeemph, unsigned char cHighpass, unsigned char cLowpass);

//Globales
unsigned char cSekunde;					//Kennzeichnungsflag fr Sekunde
unsigned char cSecTakt;					//Sekundenzhler;
unsigned char cPausenTime;				//Zhlvariable fr Pausenzeiten
unsigned char cPausenTakt;				//Arbeitsvariable fr PausenTime (sec. Takt)
unsigned char cTimerBetriebsart;		//Timer Betriebsart: 10ms Timer / 1750 Hz Timer

unsigned char BCD_Numb[11];				//BCD Ergebnisspeicher
unsigned long int uliEmpfangsfrequenz = 145775000;	//Empfangsfrequenz
unsigned long int uliSendefrequenz = 145175000;		//Sendefrequenz
unsigned long int uliAblage = 600000;				//Frequenzablage
unsigned char cAblage = 0;				//0 -> 0 kHz Ablage; 1 -> 600 kHz Ablage
unsigned char cBandbreite = 0;			//0 -> 12,5 kHz; 1 -> 25 kHz
unsigned char cSpuelch = 1;				//0 = off (1 - 8)
unsigned char cVolumen = 5;
unsigned char cPower = 0;				//0 -> 0,5W; 1 -> 1,0W

unsigned char cPrellcounter;
unsigned char cTasten_Neu;
unsigned char cTasten_Alt;
unsigned char cTastenzustand;
unsigned char cTaste_CE;
unsigned char cTaste_FCT;
unsigned char cTaste_Up;
unsigned char cTaste_Down;
unsigned char cTaste_Enter;

unsigned char LCD_String1[] = "2m-Band FM-Tx-Rx";
unsigned char LCD_String2[] = " Meinrad Goetz  ";
unsigned char LCD_String3[] = "DORJI Initialis.";
unsigned char LCD_String4[] = "   DORJO fail   ";
unsigned char LCD_String5[] = "Tx-Frequenz:    ";
unsigned char LCD_String6[] = "Rx-Frequenz:    ";
unsigned char LCD_String7[] = "Ablage: 600 kHz ";
unsigned char LCD_String8[] = "Ablage:   0 kHz ";
unsigned char LCD_String9[] = "Kanalr.: 12,5kHz";
unsigned char LCD_String10[] = "Kanalr.: 25,0kHz";
unsigned char LCD_String11[] = "Frequenzeingabe ";
unsigned char LCD_String12[] = "Pout: low (0,5W)";
unsigned char LCD_String13[] = "Pout: high (1W) ";
unsigned char LCD_String14[] = "  Fehler UART   ";

unsigned char LCD_StringSQ[] = "Squelch: x      ";
unsigned char LCD_StringVOL[] = "Volumen: x       ";

unsigned char LCD_String_Error[] = "Error           ";

unsigned char LCD_Data[16];
unsigned char LCD_String_Blank[] = "                ";//16 Zeichen fr Blank

unsigned char Receive_Buffer[20];

		
int main(void)
{
	unsigned char cState_UHF_Modul;
	unsigned char cHelp;

	Init_CPU();		//Initialisierung
	Init_Timer0();
	Init_UART();

	Init_LCD();

	LCD_Zeile1(LCD_String1);
	LCD_Zeile2(LCD_String2);
	wdt_enable(WDTO_2S);
	Pause_Timer(ANZ_VERS_NR);		//Anzeigedauer Versions Nr.
	
	LCD_Zeile1(LCD_String3);		//Init UHF Modul
	LCD_Zeile2(LCD_String4);
	Init_DRA818();
	LCD_Zeile2(Receive_Buffer);
	Pause_Timer(ANZ_VERS_NR);		//Anzeigedauer Status

	cHelp = Send_SetGroup(0, uliSendefrequenz, uliEmpfangsfrequenz, 0, cSpuelch, 0);
	if (FALSE == cHelp)
	{
		LCD_Zeile1(LCD_String14);	
		Pause_Timer(ANZ_ERROR);		//Anzeigedauer Fehlermeldung
	}

	PORTB = PORTB | (1 << PTT);		//PTT auf 1 --> Empfangsbetrieb
	cState_UHF_Modul = UHF_RX;	
	LCD_Zeile1(LCD_String6);			//Empfangsfrequenz anzeigen
	Show_Frequenz_Zeile2(uliEmpfangsfrequenz);

	cHelp = Send_FilterCommand(0,0,0);		//PRE/DE-EMPH = On; Highpass = On; Lowpass = On; (1 schaltet die Funktion aus)
	if (FALSE == cHelp)
	{
		LCD_Zeile1(LCD_String14);	
		Pause_Timer(ANZ_ERROR);		//Anzeigedauer Fehlermeldung
	}

	while (1)
	{
		if (cTaste_FCT == 1)
		{
			Funktions_Auswahl();
			cHelp = Send_SetGroup(0, uliSendefrequenz, uliEmpfangsfrequenz, 0, cSpuelch, 0);	//Gruppeneinstellung an Modul bertragen
			if (FALSE == cHelp)
			{
				LCD_Zeile1(LCD_String14);	
				Pause_Timer(ANZ_ERROR);		//Anzeigedauer Fehlermeldung
			}			
			PORTB = PORTB | (1 << PTT);		//PTT auf 1 --> Empfangsbetrieb
			cState_UHF_Modul = UHF_RX;	
			LCD_Zeile1(LCD_String6);			//Empfangsfrequenz anzeigen
			Show_Frequenz_Zeile2(uliEmpfangsfrequenz);
			cHelp = Send_VolumeCommand(cVolumen);		//Volume an Modul bertragen
			if (FALSE == cHelp)
			{
				LCD_Zeile1(LCD_String14);	
				Pause_Timer(ANZ_ERROR);				//Anzeigedauer Fehlermeldung
			}
			if (0 == cPower)						//Power einstellen
			{
				PORTB = PORTB & ~(1 << POWER);		//Power Pin auf 0 --> 0,5W Sendeleistung
			}
			else
			{
				PORTB = PORTB | (1 << POWER);		//Power Pin auf 1 --> 1,0W Sendeleistung
			}
		}

		if (cTaste_Enter == 1)
		{
			if (UHF_RX == cState_UHF_Modul)
			{
				PORTB = PORTB & ~(1 << PTT);		//PTT auf 0 --> Sendebetrieb
				cState_UHF_Modul = UHF_TX;
				LCD_Zeile1(LCD_String5);			//Sendefrequenz anzeigen
				Show_Frequenz_Zeile2(uliSendefrequenz);
			}
			if (UHF_TX == cState_UHF_Modul)
			{
				if (cTaste_CE == 1)
				{
					//1750 Hz Ton solange wie CE-Taste gedrckt ist
					TCCR0 = 0b00000000;			//Timer 0 Stop, damit Timer fr Tonerzeugung genutzt werden kann
					TCNT0 = 256 - TIMER0_1750;	//Ladewert fr 1750 Hz Ton
					cTimerBetriebsart = TON;
					TCCR0 = 0b00000010;					//Timer 0 Starten (Takt = CPU-Takt)
					cHelp = PIND;				//CE-Taste abfragen
					cHelp = cHelp & 0b00000100;
					while (cHelp == 0)
					{
						cHelp = PIND;							//CE-Taste abfragen
						cHelp = cHelp & 0b00000100;
						wdt_reset();
					}
					TCCR0 = 0b00000000;				//Timer 0 Stop, und auf 10ms Timer zurckschalten
					TCNT0 = 256 - TIMER0_10MS;		//Ladewert fr 10ms
					cTimerBetriebsart = TIMER;
					TCCR0 = 0b00000100;				//Timer 0 wieder Starten (Takt = CPU-Takt/256)
					cSecTakt = 0;					//Sekundentakt = 0
					cSekunde = 0;
					PORTB = PORTB & ~(1 << TON_1750);	//Port 1750 Hz Ton auf 0 (Einschwingverhalten)
					while (cTaste_CE == 1)
					{
						cHelp++;				//Warten bis CE Taste losgelassen (entprellt) ist
					}
				}
			}
		}
		else
		{
			if (UHF_TX == cState_UHF_Modul)
			{
				PORTB = PORTB | (1 << PTT);			//PTT auf 1 --> Empfangsbetrieb
				cState_UHF_Modul = UHF_RX;
				LCD_Zeile1(LCD_String6);			//Empfangsfrequenz anzeigen
				Show_Frequenz_Zeile2(uliEmpfangsfrequenz);
			}
		}

		wdt_reset();
	}
}	


/****************************************************************
*																*
*	Initialisierung CPU 										*
*																*
*****************************************************************/
void Init_CPU(void)
{
	DDRB = 0x07;		//PB7:In   PB6:In   PB5:Out  PB4:Out  PB3:In  PB2:Out  PB1:Out  PB0:Out
	PORTB = 0x38;		//PB7:k.PU PB6:k.PU PB5:PU   PB4:PU   PB3:PU  PB2:0    PB1:1    PB0:0

	DDRC = 0x3F;		//x   PC6:IN   PC5:Out  PC4:Out  PC3:Out  PC2:Out  PC1:Out  PC0:Out
	PORTC = 0x00;		//x   PC6:k.PU PC5:0    PC4:0    PC3:0    PC2:0    PC1:0    PC0:0 

	DDRD = 0xFF;		//PD7:IN PD6:IN PD5:IN PD4:IN PD3:IN PD2:IN PD1:Out PD0:IN
	PORTD = 0xFF; 		//PD7:PU PD6:PU PD5:PU PD4:PU PD3:PU PD2:PU PD1:1   PD0:PU

	cPrellcounter = 0;
	cTasten_Neu = PIND;				//Tastatur initialisieren
	cTasten_Neu = cTasten_Neu & TASTENMASKE;
	cTasten_Alt = cTasten_Neu;
	cTastenzustand = 0b11111100;	//keine Taste gedrckt
	cTaste_CE = 0;
	cTaste_FCT = 0;
	cTaste_Up = 0;
	cTaste_Down = 0;
	cTaste_Enter = 0;

	wdt_enable(WDTO_2S);
	sei();						//Interrupts freigeben
}


void Init_DRA818 (void)
{
	unsigned char Init_String[] = "AT+DMOCONNECT";
	
	Send_String (&Init_String[0], 13);
	Send_CR_LF ();
	Receive_String (&Receive_Buffer[0], 15);

	Receive_Buffer[13] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[14] = ' ';	//(wegen Anzeige) lschen.
	Receive_Buffer[15] = ' ';
}


void Funktions_Auswahl(void)
{
	unsigned char cFunktionsNummer = 0;		//Frequenz (0); Ablage (1); Kanalraster (2); Squelch (3); Volumen (4); Power (5);
	unsigned char cNewState = 1;

	while (cTaste_Enter == 0)
	{
		if (cTaste_Up == 1)
		{
			cFunktionsNummer++;
			if (cFunktionsNummer == 6)
			{
				cFunktionsNummer = 0;
			}
			cNewState = 1;
		}
		if (cTaste_Down == 1)
		{
			cFunktionsNummer--;
			if (cFunktionsNummer == 0xFF)
			{
				cFunktionsNummer = 5;
			}	
			cNewState = 1;
		}

		if (cNewState == 1)
		{
			switch (cFunktionsNummer)
			{
				case 0:							//Frequenz
					LCD_Zeile1(LCD_String11);
					Show_Frequenz_Zeile2(uliEmpfangsfrequenz);
					break;

				case 1:							//Ablage
					if (cAblage == 0)
					{
						LCD_Zeile1(LCD_String7);
					}
					else
					{
						LCD_Zeile1(LCD_String8);
					}
					LCD_Zeile2(LCD_String_Blank);
					break;

				case 2:							//Kanalraster
					if (cBandbreite == 0)
					{
						LCD_Zeile1(LCD_String9);
					}
					else
					{
						LCD_Zeile1(LCD_String10);
					}
					LCD_Zeile2(LCD_String_Blank);
					break;

				case 3:							//Squelch
					LCD_StringSQ[9] = cSpuelch + 0x30;		//Squelch value in ASCII
					LCD_Zeile1(LCD_StringSQ);
					LCD_Zeile2(LCD_String_Blank);
					break;
				
				case 4:
					LCD_StringVOL[9] = cVolumen + 0x30;		//Volume value in ASCII
					LCD_Zeile1(LCD_StringVOL);
					LCD_Zeile2(LCD_String_Blank);					
					break;

				case 5:
					if (cPower == 0)
					{
						LCD_Zeile1(LCD_String12);
					}
					else
					{
						LCD_Zeile1(LCD_String13);
					}
					LCD_Zeile2(LCD_String_Blank);				
					break;


				default:
					LCD_Zeile1(LCD_String_Error);
					LCD_Zeile2(LCD_String_Blank);					
					break;
			}
			Wait_Tasten_Open();
			cNewState = 0;
		}
	}
	Wait_Tasten_Open();	

	//Eine Funktionsgruppe ist ausgewhlt
	while (cTaste_Enter == 0)
	{
		if ((cTaste_Up == 1) || (cTaste_Down == 1) || (cTaste_CE ==1))
		{
			cNewState = 1;
		}
		if (1 == cNewState)
		{
			switch (cFunktionsNummer)
			{
				case 0:							//Frequenz
					if (cTaste_CE == 1)
					{
						uliEmpfangsfrequenz = uliEmpfangsfrequenz + 100000;
						if (uliEmpfangsfrequenz > 146000000)
						{
							uliEmpfangsfrequenz = 144000000;
						}
					}


					if (cTaste_Up == 1)
					{
						if (cBandbreite == 0)
						{
							uliEmpfangsfrequenz = uliEmpfangsfrequenz + 12500;
						}
						else
						{
							uliEmpfangsfrequenz = uliEmpfangsfrequenz + 25000;
						}
						if (uliEmpfangsfrequenz > 146000000)
						{
							uliEmpfangsfrequenz = 146000000;
						}
					}
					if (cTaste_Down == 1)
					{
						if (cBandbreite == 0)
						{
							uliEmpfangsfrequenz = uliEmpfangsfrequenz - 12500;
						}
						else
						{
							uliEmpfangsfrequenz = uliEmpfangsfrequenz - 25000;
						}
						if (uliEmpfangsfrequenz < 144000000)
						{
							uliEmpfangsfrequenz = 144000000;
						}
					}
					LCD_Zeile1(LCD_String11);
					Show_Frequenz_Zeile2(uliEmpfangsfrequenz);
					break;

				case 1:							//Ablage
					if ((cTaste_Up == 1) || (cTaste_Down == 1))
					{
						cAblage++;
						cAblage = cAblage & 0x01;
					}
					if (cAblage == 0)
					{
						LCD_Zeile1(LCD_String7);
					}
					else
					{
						LCD_Zeile1(LCD_String8);
					}
					LCD_Zeile2(LCD_String_Blank);
					break;

				case 2:							//Kanalraster
					if ((cTaste_Up == 1) || (cTaste_Down == 1))
					{
						cBandbreite++;
						cBandbreite = cBandbreite & 0x01;
					}
					if (cBandbreite == 0)
					{
						LCD_Zeile1(LCD_String9);
					}
					else
					{
						LCD_Zeile1(LCD_String10);
					}
					LCD_Zeile2(LCD_String_Blank);
					break;

				case 3:							//Squelch
					if (cTaste_Up == 1)
					{
						cSpuelch++;
						if (cSpuelch > 8)
						{
							cSpuelch = 8;
						}
					}
					if (cTaste_Down == 1)
					{
						cSpuelch--;
						if (cSpuelch == 0xFF)
						{
							cSpuelch = 0;
						}
					}
					LCD_StringSQ[9] = cSpuelch + 0x30;		//Squelch value in ASCII
					LCD_Zeile1(LCD_StringSQ);
					LCD_Zeile2(LCD_String_Blank);
					break;
				
				case 4:
					if (cTaste_Up == 1)
					{
						cVolumen++;
						if (cVolumen > 8)
						{
							cVolumen = 8;
						}
					}
					if (cTaste_Down == 1)
					{
						cVolumen--;
						if (cVolumen == 0)
						{
							cVolumen = 1;
						}
					}
					LCD_StringVOL[9] = cVolumen + 0x30;		//Squelch value in ASCII
					LCD_Zeile1(LCD_StringVOL);
					LCD_Zeile2(LCD_String_Blank);
					break;
					
				case 5:
					if ((cTaste_Up == 1) || (cTaste_Down == 1))
					{
						cPower++;
						cPower = cPower & 0x01;
					}
					if (cPower == 0)
					{
						LCD_Zeile1(LCD_String12);
					}
					else
					{
						LCD_Zeile1(LCD_String13);
					}
					LCD_Zeile2(LCD_String_Blank);
			}
			Wait_Tasten_Open();
			cNewState = 0;
		}
	}
	if (cAblage == 0)
	{
		uliSendefrequenz = uliEmpfangsfrequenz - 600000;
	}
	else
	{
		uliSendefrequenz = uliEmpfangsfrequenz;
	}
	Wait_Tasten_Open();		//Warten bis Entertaste geffnet ist.
}



/****************************************************************
*	Unsigned Integer auf BCD  Convertierung						*
*																*
*	Der bergebene unsigned long Integerwert wird in eine       *
*	10-stellige BCD Zahl gewandelt.								*
*	Das Ergebnis steht in BCD_Numb[] 							*
****************************************************************/
void Convert_US_Int_Bcd(unsigned long int Zahl)           //10-stellig
{
	BCD_Numb[10] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[9] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[8] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[7] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[6] = Zahl % 10;
    Zahl = Zahl/10;
	BCD_Numb[5] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[4] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[3] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[2] = Zahl % 10;
    Zahl = Zahl/10;
    BCD_Numb[1] = Zahl % 10;
}      




/****************************************************************
*	BCD auf ASCII Convertierung									*
*																*
*	Die in BCD_Numb[] stehende 10-stellige BCD Zahl wird auf 	*
*	ASCII gewandelt und wieder in BCD_Numb[]gespeichert.		*
*   Das auf BCD_Numb[0] stehende Vorzeichen ist schon auf ASCII.*
****************************************************************/
void Convert_BCD_ASCII(void)
{
	unsigned char i;

	for (i = 1; i < 11; i++)
	{
		BCD_Numb[i] = BCD_Numb[i] + 0x30;
	}
}





/****************************************************************
*	LCD Funktionen												*
****************************************************************/

/*	Init LCD
	Programmteil zur Initialisierung des LCD Port und des LCD */
void Init_LCD (void)
{
	Wait(6000);		//Wartezeit fr stabile Betriebsspannung  (min. 15ms)
	WR_COMAND(0b00000011);
	Wait(1000);		//min. 4,1ms
	WR_COMAND(0b00000011);
	Wait(200);		//min. 100s
	WR_COMAND(0b00000011);
	Wait(1000);
	WR_COMAND(0b00000010);	//Function SET
	Wait(1000);
	WR_COMAND(0b00000010);	//set interface to 4-Bit
	Wait(1000);
	WR_COMAND(0b00001000);	//2 - lines, 5x7 Font
	Wait(1000);
	WR_COMAND(0b00000000);	//Display OFF (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00001000);	//Display OFF (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Display Clear (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00000001);	//Display Clear (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Entry Mode set (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00000110);	//Entry Mode set (2. Nibble)
	Wait(1000);
	WR_COMAND(0b00000000);	//Switch Display ON (1. Nibble)
	Wait(1000);
	WR_COMAND(0b00001110);	//Switch Display ON (2. Niblle)
	Wait(1000);
}


/*	LCD Ausgabe auf Zeile 1	*/
void LCD_Zeile1(unsigned char *pLCD)
{
	unsigned char i;
	unsigned char j;

	WR_COMAND(0b00001000);	//Set Adresse 0x80 fr 1. Zeile (low Nibble)
	WR_COMAND(0b00000000);	//Set Adresse 0x80 fr 1. Zeile (high Nibble)
	Wait(500);
	for (i = 0; i < 16; i++)
	{
		j = *pLCD;
		WR_DATA(j);
		pLCD++;
	}
}


/*	LCD Ausgabe auf Zeile 2	*/
void LCD_Zeile2(unsigned char *pLCD)
{
	unsigned char i;
	unsigned char j;

	WR_COMAND(0b00001100);	//Set Adresse 0xC0 fr 1. Zeile (low Nibble)
	WR_COMAND(0b00000000);	//Set Adresse 0xC0 fr 1. Zeile (high Nibble)
	Wait(500);
	for (i = 0; i < 16; i++)
	{
		j = *pLCD;
		WR_DATA(j);
		pLCD++;
	}
}


/* Frequenzanzeige in Zeile 2 */
void Show_Frequenz_Zeile2(unsigned long int uliFrequenz)
{
	Convert_US_Int_Bcd(uliFrequenz);
	Convert_BCD_ASCII();
	LCD_Data[0] = BCD_Numb[2];
	LCD_Data[1] = BCD_Numb[3];
	LCD_Data[2] = BCD_Numb[4];
	LCD_Data[3] = ',';
	LCD_Data[4] = BCD_Numb[5];
	LCD_Data[5] = BCD_Numb[6];
	LCD_Data[6] = BCD_Numb[7];
	LCD_Data[7] = BCD_Numb[8];
	LCD_Data[8] = BCD_Numb[9];
	LCD_Data[9] = BCD_Numb[10];
	LCD_Data[10] = ' ';
	LCD_Data[11] = 'M';
	LCD_Data[12] = 'H';
	LCD_Data[13] = 'z';
	LCD_Data[14] = ' ';
	LCD_Data[15] = ' ';
	LCD_Zeile2(LCD_Data);
}


/* 	Write Commando an LCD
	Daten auf PortC.0 - PortC.3, RS auf PortC.4, E auf PortC.5 */
void WR_COMAND(unsigned char i)
{
	i = i & 0x0F;		//Daten isolieren (RS und E auf 0 schalten)
	PORTC = i;
	Wait(10);
	i = i | 0b00100000;	//E Signal setzen
	PORTC = i;
	Wait(10);
	i = i & 0x0F;		//E Signal zurcknehmen
	PORTC = i;	
	Wait(10);
}


/*	Write Data an LCD
	Daten auf PortC.0 - PortC.3, RS auf PortC.4, E auf PortC.5 */
void WR_DATA(unsigned char i)
{
	unsigned char j;

	j = i;
	j = j & 0xF0;		//oberes Nibble zuerst senden
	j = j>>4;
	j = j | 0b00010000;	//RS fr Daten auf 1 setzen
	PORTC = j;
	Wait(10);
	j = j | 0b00100000;	//E Signal setzen
	PORTC = j;
	Wait(10);
	j = j & 0x1F;		//E Signal zurcknehmen
	PORTC = j;
	Wait(10);
	j = i;				//unters Nibble senden
	j = j & 0x0F;
	j = j | 0b00010000;	//RS fr Daten auf 1 setzen
	PORTC = j;
	Wait(10);
	j = j | 0b00100000;	//E Signal setzen
	PORTC = j;
	Wait(10);
	j = j & 0x1F;		//E Signal zurcknehmen
	PORTC = j;
	Wait(10);
}




/*	Initialisierung Timer 0	auf 10ms	*/
void Init_Timer0(void)
{
	cTimerBetriebsart = TIMER;		//Timer 0 als 10 ms Timer
	TCCR0 = 0b00000000;			//Timer 0 Stop
	TIMSK = (1 << TOIE0);		//Timer 0 Interrupt freigeben
	TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms
	TCCR0 = 0b00000100;			//Timer 0 Starten (Takt = CPU-Takt/256)
	cSecTakt = 0;				//Sekundentakt = 0
	cSekunde = 0;
}


/* Funktion zur Erzeugung einer Wartezeit */
/* 2MHz:	i =    1 --> 120s
			i =   10 --> 1,2ms
			i =  100 --> 12ms
			i = 1000 --> 120ms            */
void Wait(unsigned int i)
{
	unsigned char j;
	while (i != 0)
	{
		j = 10;
		while (j != 0)
		{
			j--;
		}
		i--;
	}
}



/* Funktion zur Erzeugung einer Wartezeit im Sekundentakt
   maximale Wartezeit: 255 sec. Steuerung ber Timer 0
   Wenn inerhalb der Wartezeit eine beliebige Taste gedrckt wird
   wird die Wartezeit abgebrochen. */
void Pause_Timer (unsigned char Zeit)
{
	unsigned char Tastenzustand = 0b11111100;

	cPausenTakt = 0;				//Pausentaktsteuerung synchronisieren
	cPausenTime = Zeit;			//Zeit bernehmen
	while (cPausenTime != 0 && Tastenzustand == 0b11111100)
	{
		//Warten bis Zeit abgelaufen ist
		Tastenzustand = PIND;
		Tastenzustand = Tastenzustand & 0b11111100;
		wdt_reset();
	}
}


/* Interrupt Service Routinen	*/

ISR(TIMER0_OVF_vect)
{
	if (TON == cTimerBetriebsart)
	{
		TCNT0 = 256 - TIMER0_1750;	//Ladewert fr 1750 Hz
		PORTB^=(1<<TON_1750);
	}
	else
	{
		TCNT0 = 256 - TIMER0_10MS;	//Ladewert fr 10ms

		wdt_reset();

		cSecTakt++;
		if (cSecTakt == 60)
		{
			cSekunde = 1;			//Sekundentakt
			cSecTakt = 0;
		}

		cPausenTakt++;
		if (cPausenTakt == 60)
		{
			cPausenTakt = 0;			//Pausentakt Steuerung
			cPausenTime--;
		}

		cTasten_Neu = PIND;			//Tastatur einlesen
		cTasten_Neu = cTasten_Neu & TASTENMASKE;
		if (cTasten_Neu == cTasten_Alt)
		{
			cPrellcounter++;
			if(cPrellcounter == ENTPRELLZEIT)
			{
				cPrellcounter = 0;
				cTastenzustand = cTasten_Neu;
				if((cTastenzustand & 0b00000100) == 0)
				{
					cTaste_CE = 1;
				}
				else
				{
					cTaste_CE = 0;
				}
				if((cTastenzustand & 0b00001000) == 0)
				{
					cTaste_FCT = 1;
				}
				else
				{
					cTaste_FCT = 0;
				}
				if((cTastenzustand & 0b00010000) == 0)
				{
					cTaste_Up = 1;
				}
				else
				{
					cTaste_Up = 0;
				}
				if((cTastenzustand & 0b00100000) == 0)
				{
					cTaste_Down = 1;
				}
				else
				{
					cTaste_Down = 0;
				}
				if((cTastenzustand & 0b01000000) == 0)
				{
					cTaste_Enter = 1;
				}
				else
				{
					cTaste_Enter = 0;
				}
			}
		}
		else
		{
			cTasten_Alt = cTasten_Neu;
			cPrellcounter = 0;
		}
	}
}



void Wait_Tasten_Open(void)
{
	while (cTastenzustand != 0b11111100)
	{
		//auf loslassen der Tasten warten
		wdt_reset();
	}
}


/********************************************************************
*																	*
*	UART Funktionen													*
*																	*
********************************************************************/
void Init_UART(void)
{
/* Set baud rate to 9600 Baud */
UBRRH = 0;
UBRRL = 23;
/*  11 -> 19,2 kBaud
	23 -> 9600 Baud
	47 -> 4800 Baud 
   	95 -> 2400 Baud */
;
/* Enable receiver and transmitter */
UCSRB = (1<<RXEN)|(1<<TXEN);

/* Enable only transmitter */
//UCSRB = (1<<TXEN);

/* Set frame format: 8data, 1stop bit */
UCSRC = (1<<URSEL)|(3<<UCSZ0);
}


void Send_UART (unsigned char cData)
{
	// Wait if a byte is being transmitted
	while((UCSRA&(1<<UDRE)) == 0);

	// Transmit data
	UDR = cData;
	wdt_reset();
}

void Send_CR_LF (void)
{
	Send_UART (0x0D);		//CR
	Send_UART (0x0A);		//LF
}


void Send_String (unsigned char *pSource, unsigned char cAnzahl)
{
	unsigned char i;

	while (cAnzahl != 0)
	{
		i = *pSource;
		Send_UART(i);
		pSource++;
		cAnzahl--;
	}
}

unsigned char Receive_UART (void)
{
	/* Wait for data to be received */
	while ( !(UCSRA & (1<<RXC)) );

	wdt_reset();
	/* Read data from buffer */
	return UDR;
}

void Receive_String (unsigned char *pDestination, unsigned char cAnzahl)
{
	unsigned char i;

	while (cAnzahl != 0)
	{
		i = Receive_UART ();
		*pDestination = i;
		pDestination++;
		cAnzahl--;
	}
}


unsigned char Send_SetGroup(unsigned char cBW, unsigned long int uliTx_Frequenz, unsigned long int uliRx_Frequenz,
				   unsigned int Tx_CTCSS, unsigned char cSqu, unsigned int Rx_CTCSS)
{
	//AT+DMOSETGROUP=GBW,TFV, RFV,Tx_CTCSS,SQ,Rx_CTCSS<CR><LF>
	unsigned char Init_String[] = "AT+DMOSETGROUP=";

	Send_String (&Init_String[0], 15);		//Sendebefehl Group
	Send_UART (cBW + 0x30);					//Bandbreite (ASCII)
	Send_UART (',');
	Convert_US_Int_Bcd(uliTx_Frequenz);		//Sendefrequenz
	Convert_BCD_ASCII();
	Send_UART (BCD_Numb[2]);	
	Send_UART (BCD_Numb[3]);	
	Send_UART (BCD_Numb[4]);	
	Send_UART ('.');	
	Send_UART (BCD_Numb[5]);	
	Send_UART (BCD_Numb[6]);	
	Send_UART (BCD_Numb[7]);	
	Send_UART (BCD_Numb[8]);
	Send_UART (',');

	Convert_US_Int_Bcd(uliRx_Frequenz);		//Emfpangsfrequenz
	Convert_BCD_ASCII();
	Send_UART (BCD_Numb[2]);	
	Send_UART (BCD_Numb[3]);	
	Send_UART (BCD_Numb[4]);	
	Send_UART ('.');	
	Send_UART (BCD_Numb[5]);	
	Send_UART (BCD_Numb[6]);	
	Send_UART (BCD_Numb[7]);	
	Send_UART (BCD_Numb[8]);
	Send_UART (',');

	Send_UART ('0');						//Tx_CTCSS abgeschaltet
	Send_UART ('0');
	Send_UART ('0');
	Send_UART ('0');
	Send_UART (',');

	Send_UART (cSqu + 0x30);				//Squelch als ASCII
	Send_UART (',');

	Send_UART ('0');						//Rx_CTCSS abgeschaltet
	Send_UART ('0');
	Send_UART ('0');
	Send_UART ('0');

	Send_CR_LF ();

	Receive_String (&Receive_Buffer[0], 16);
	Receive_Buffer[14] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[15] = ' ';	//(wegen Anzeige) lschen.
	//LCD_Zeile2(Receive_Buffer);
	//Pause_Timer(ANZ_VERS_NR);		//Anzeigedauer Status	+DMOSETGROUP:X<CR><LF>

	if ('0' == Receive_Buffer[13])
		return TRUE;
	else
		return FALSE;
}


unsigned char Send_VolumeCommand(unsigned char cVol)
{
	//AT+DMOSETVOLUME=x <CR><LF>
	unsigned char Volume_String[] = "AT+DMOSETVOLUME=";
	
	Send_String (&Volume_String[0], 16);		//Sendebefehl Volume
	Send_UART (cVol + 0x30);					//Volume als ASCII
	Send_CR_LF ();

	Receive_String (&Receive_Buffer[0], 17);
	Receive_Buffer[15] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[16] = ' ';	//(wegen Anzeige) lschen.
	//LCD_Zeile2(Receive_Buffer);
	//Pause_Timer(ANZ_VERS_NR);		//Anzeigedauer Rckmeldung	+DMOSETVOLUME:X<CR><LF>
	if ('0' == Receive_Buffer[14])
		return TRUE;
	else
		return FALSE;

}
	

unsigned char Send_FilterCommand(unsigned char cDeemph, unsigned char cHighpass, unsigned char cLowpass)
{
	//AT+SETFILTER=PRE/DE-EMPH,Highpass,Lowpass <CR><LF>
	unsigned char Filter_String[] = "AT+SETFILTER=";
	
	Send_String (&Filter_String[0], 13);		//Sendebefehl Filter
	Send_UART (cDeemph + 0x30);					//Deemphasis als ASCII
	Send_UART (',');
	Send_UART (cHighpass + 0x30);				//Highpass als ASCII
	Send_UART (',');
	Send_UART (cLowpass + 0x30);				//Lowpass als ASCII
	Send_CR_LF ();
	
	Receive_String (&Receive_Buffer[0], 17);
	Receive_Buffer[15] = ' ';	//CR LF aus Empfangspuffer
	Receive_Buffer[16] = ' ';	//(wegen Anzeige) lschen.
	//LCD_Zeile2(Receive_Buffer);
	//Pause_Timer(ANZ_VERS_NR);		//Anzeigedauer Rckmeldung	+DMOSETFILTER:X<CR><LF>
	if ('0' == Receive_Buffer[14])
		return TRUE;
	else
		return FALSE;
}
